<?php
// generate_word.php

// Ensure Composer's autoloader is included using a robust path.
require_once __DIR__ . '/vendor/autoload.php';

// Use PHPWord classes
use PhpOffice\PhpWord\TemplateProcessor;
// Removed unused alias for non-existent class PhpOffice\PhpWord\Element\Html

// --- Step 1: Define paths and check for required files ---
$html_file_path = __DIR__ . '/sample.html';
$templateFile = __DIR__ . '/template.docx';

if (!file_exists($html_file_path)) {
    error_log("Error: sample.html not found at path: " . $html_file_path);
    die("Error: The required HTML input file (sample.html) was not found. Please ensure it exists in the script's directory.");
}
if (!file_exists($templateFile)) {
    error_log("Error: Word template file '{$templateFile}' not found.");
    die("Error: The Word template file ('template.docx') was not found. Please create it with placeholders like \${title}, \${author}, \${report_date}, and \${content_block}.");
}

// --- Step 2: Extract Content from HTML ---
$html_content_full = file_get_contents($html_file_path);
if ($html_content_full === false) {
    error_log("Error: Could not read sample.html from path: " . $html_file_path);
    die("Error: Could not read sample.html. Please check file permissions and path.");
}

// Using DOMDocument to parse the HTML and extract specific parts.
$dom = new DOMDocument();
// Suppress errors for potentially malformed HTML and avoid adding implicit <html><body> tags.
@$dom->loadHTML($html_content_full, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD | LIBXML_NOWARNING | LIBXML_NOERROR);

$titleElement = $dom->getElementById('main-title');
$contentElement = $dom->getElementById('content');

$documentTitle = $titleElement ? trim($titleElement->nodeValue) : 'Default Document Title';

$documentContentHTML = '';
if ($contentElement) {
    foreach ($contentElement->childNodes as $child) {
        $documentContentHTML .= $dom->saveHTML($child);
    }
} else {
    $documentContentHTML = '<p>Default content. The element with ID "content" was not found in sample.html.</p>';
    error_log("Warning: Element with ID 'content' not found in sample.html. Using default content for Word document.");
}

// --- Step 3: Process Word Template ---
try {
    $templateProcessor = new TemplateProcessor($templateFile);

    // Replace simple text placeholders. htmlspecialchars is good for preventing XML issues.
    $templateProcessor->setValue('title', htmlspecialchars($documentTitle, ENT_QUOTES, 'UTF-8'));
    $templateProcessor->setValue('author', 'Robin Metcalfe (Solarise)');
    $templateProcessor->setValue('report_date', date('F j, Y, g:i a'));

    // Replace content block with HTML.
    // PHPWord's HTML import has limitations with complex CSS/layouts.
    // For best results, keep the HTML in 'content_block' relatively simple (paragraphs, lists, basic tables, text formatting).
    if (!empty($documentContentHTML)) {
        // PHPWord does not support direct HTML objects. Use addHtml or plain text as a workaround.
        $templateProcessor->setValue('content_block', strip_tags($documentContentHTML));
    } else {
        $templateProcessor->setValue('content_block', 'No content could be extracted from the HTML.');
    }

    // --- Step 4: Output the Document ---
    $outputFilename = 'generated_word_from_sample_' . date('Ymd_His') . '.docx';

    header("Content-Description: File Transfer");
    header('Content-Disposition: attachment; filename="' . $outputFilename . '"');
    header('Content-Type: application/vnd.openxmlformats-officedocument.wordprocessingml.document');
    header('Content-Transfer-Encoding: binary');
    header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
    header('Expires: 0');
    header('Pragma: public');

    // Save the processed template to PHP's output stream
    $templateProcessor->saveAs('php://output');
    exit;

} catch (\PhpOffice\PhpWord\Exception\Exception $e) {
    error_log("PHPWord Exception: " . $e->getMessage() . "\n" . $e->getTraceAsString());
    die("Error generating Word document (PHPWord Exception): " . htmlspecialchars($e->getMessage()));
} catch (Exception $e) {
    error_log("General Error in Word Generation: " . $e->getMessage() . "\n" . $e->getTraceAsString());
    die("Error generating Word document (General Error): " . htmlspecialchars($e->getMessage()));
}
?>
