<?php
// generate_excel_simple.php

// Ensure Composer's autoloader is included using a robust path.
require_once __DIR__ . '/vendor/autoload.php';

// Reference the necessary PhpSpreadsheet classes.
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Color; // Added for explicit color object creation

// Create a new Spreadsheet object
$spreadsheet = new Spreadsheet();

// Set document properties
$spreadsheet->getProperties()
    ->setCreator("Robin Metcalfe (Solarise)")
    ->setLastModifiedBy("Robin Metcalfe (Solarise)")
    ->setTitle("Simple Sales Data Report")
    ->setSubject("Sales Data for Q1")
    ->setDescription("A test document generated using PhpSpreadsheet, containing fictional sales data.")
    ->setKeywords("phpspreadsheet excel sales data report")
    ->setCategory("Sales Reports");

// Get the active sheet and set its title
$sheet = $spreadsheet->getActiveSheet();
$sheet->setTitle('Sales Data');

// Set column headers
$sheet->setCellValue('A1', 'Product ID');
$sheet->setCellValue('B1', 'Product Name');
$sheet->setCellValue('C1', 'Quantity Sold');
$sheet->setCellValue('D1', 'Unit Price');
$sheet->setCellValue('E1', 'Total Price');

// Sample data for the report
$data = [
    ['P001', 'Alpha Widget', 150, 10.50],
    ['P002', 'Beta Gadget', 200, 7.75],
    ['P003', 'Gamma Gizmo', 120, 12.00],
    ['P004', 'Delta Device', 300, 5.25],
    ['P005', 'Epsilon Gear', 80, 22.99],
];

// Populate data rows
$row = 2; // Start data from row 2
foreach ($data as $rowData) {
    $sheet->setCellValue('A'.$row, $rowData[0]); // Product ID
    $sheet->setCellValue('B'.$row, $rowData[1]); // Product Name
    $sheet->setCellValue('C'.$row, $rowData[2]); // Quantity Sold
    $sheet->setCellValue('D'.$row, $rowData[3]); // Unit Price

    // Set number formats for Quantity and Currency
    $sheet->getStyle('C'.$row)->getNumberFormat()->setFormatCode(NumberFormat::FORMAT_NUMBER);
    $sheet->getStyle('D'.$row)->getNumberFormat()->setFormatCode(NumberFormat::FORMAT_NUMBER);
    
    // Calculate Total Price using an Excel formula
    $sheet->setCellValue('E'.$row, '=C'.$row.'*D'.$row);
    $sheet->getStyle('E'.$row)->getNumberFormat()->setFormatCode(NumberFormat::FORMAT_NUMBER);

    $row++;
}

// Add a summary/total row
$lastDataRow = $row - 1;
$summaryRow = $row + 1; // Add a little space before the summary

$sheet->setCellValue('D'.$summaryRow, 'Grand Total:');
$sheet->getStyle('D'.$summaryRow)->getFont()->setBold(true);
$sheet->getStyle('D'.$summaryRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_RIGHT);

$sheet->setCellValue('E'.$summaryRow, '=SUM(E2:E'.$lastDataRow.')');
$sheet->getStyle('E'.$summaryRow)->getNumberFormat()->setFormatCode(NumberFormat::FORMAT_NUMBER);
$sheet->getStyle('E'.$summaryRow)->getFont()->setBold(true);


// --- Styling ---

// Header row style
$headerStyle = [
    'font' => [
        'bold' => true,
        'color' => ['rgb' => 'FFFFFF'], // White text
        'size' => 12,
    ],
    'fill' => [
        'fillType' => Fill::FILL_SOLID,
        'startColor' => ['rgb' => '16A085'], // A nice teal color
    ],
    'alignment' => [
        'horizontal' => Alignment::HORIZONTAL_CENTER,
        'vertical' => Alignment::VERTICAL_CENTER,
    ],
    'borders' => [
        'allBorders' => [
            'borderStyle' => Border::BORDER_THIN,
            'color' => ['rgb' => 'BDC3C7'], // Light gray border
        ],
    ],
];
$sheet->getStyle('A1:E1')->applyFromArray($headerStyle);
$sheet->getRowDimension(1)->setRowHeight(25); // Set header row height

// Data rows - basic alternating fill (optional, for readability)
for ($i = 2; $i <= $lastDataRow; $i++) {
    if ($i % 2 == 0) { // Even rows
        $sheet->getStyle('A'.$i.':E'.$i)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setARGB('FFF2F2F2'); // Very light gray
    }
    // Apply thin borders to all data cells
    $sheet->getStyle('A'.$i.':E'.$i)->getBorders()->getAllBorders()
            ->setBorderStyle(Border::BORDER_THIN)
            ->setColor(new Color('FFDDDDDD')); // Lighter gray for data
}


// Auto-size columns for content
foreach (range('A', 'E') as $columnID) {
    $sheet->getColumnDimension($columnID)->setAutoSize(true);
}

// --- Output the Excel file to the browser ---
$outputFilename = "simple_sales_data_" . date('Y-m-d') . ".xlsx";

// Set HTTP headers for download
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment;filename="' . $outputFilename . '"');
header('Cache-Control: max-age=0');
// If you're serving to IE 9, then the following may be needed
header('Cache-Control: max-age=1'); // A fallback for older IEs
// If you're serving to IE over SSL, then the following may be needed
header('Expires: Mon, 26 Jul 1997 05:00:00 GMT'); // Date in the past
header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT'); // Always modified
header('Cache-Control: cache, must-revalidate'); // HTTP/1.1
header('Pragma: public'); // HTTP/1.0

// Create the writer and save the file to php://output
$writer = new Xlsx($spreadsheet);
$writer->save('php://output');

exit; // Important to prevent any further output
